#include <iostream>
#include <vector>
#include <cmath>
#include <numeric> // Для std::iota

// Максимальное значение для r по условию: 10^6
const int MAX_R = 1000000;

// spf[i] хранит наименьший простой делитель числа i.
// Размер MAX_R + 1 для включения MAX_R.
std::vector<int> spf(MAX_R + 1);

// is_odd_power[i] хранит true, если i - нечётно-степенное число, false в противном случае.
// Размер MAX_R + 1 для включения MAX_R.
std::vector<bool> is_odd_power(MAX_R + 1);

// Функция для предварительного вычисления наименьших простых делителей с использованием решета.
void sieve_spf() {
    // Инициализируем spf[i] = i для всех i.
    // std::iota заполняет диапазон [first, last) последовательно возрастающими значениями, начиная с value.
    std::iota(spf.begin(), spf.end(), 0); 

    // Итерируем от 2 до корня из MAX_R.
    // Если spf[i] == i, то i - простое число.
    for (int i = 2; i * i <= MAX_R; ++i) {
        if (spf[i] == i) { // i - простое число
            // Для всех кратных i (начиная с i*i) до MAX_R,
            // если их наименьший простой делитель еще не найден (т.е. он все еще равен самому себе),
            // то i является их наименьшим простым делителем.
            for (int j = i * i; j <= MAX_R; j += i) {
                if (spf[j] == j) { // если spf[j] еще не помечен меньшим простым числом
                    spf[j] = i;
                }
            }
        }
    }
}

// Функция для предварительного определения, является ли каждое число до r_max нечётно-степенным.
void precompute_odd_powers(int r_max) {
    // 0 и 1 не являются нечётно-степенными числами по определению (натуральные числа начинаются с 1 или 2,
    // и для 1 все простые делители имеют степень 0, которая чётная).
    is_odd_power[0] = false; 
    is_odd_power[1] = false;

    // Итерируем по каждому числу от 2 до r_max.
    for (int i = 2; i <= r_max; ++i) {
        int num = i;
        bool current_is_odd_power = true;
        
        // Факторизуем num, используя предварительно вычисленные значения spf.
        while (num > 1) {
            int p = spf[num]; // Наименьший простой делитель текущего num
            int count = 0;     // Степень простого p в разложении i
            
            // Считаем, сколько раз p делит num.
            while (num % p == 0) {
                count++;
                num /= p;
            }
            
            // Если степень (count) четная, то i НЕ является нечётно-степенным числом.
            if (count % 2 == 0) {                current_is_odd_power = false;
                break; // Нет необходимости проверять другие простые делители для этого числа i.
            }
        }
        is_odd_power[i] = current_is_odd_power;
    }
}

int main() {
    // Оптимизируем стандартные потоки C++ для более быстрого ввода/вывода.
    std::ios_base::sync_with_stdio(false);
    std::cin.tie(NULL);

    int m, l, r;
    std::cin >> m;
    std::cin >> l >> r;

    // Выполняем решето для SPF до MAX_R один раз.
    sieve_spf();

    // Определяем нечётно-степенные числа до r.
    precompute_odd_powers(r);

    // Ищем первую последовательность из m подряд идущих нечётно-степенных чисел в [l, r].
    int consecutive_count = 0; // Считает подряд идущие нечётно-степенные числа.
    int first_in_sequence = -1; // Хранит начальное число найденной последовательности.

    // Итерируем от l до r для поиска последовательности.
    for (int i = l; i <= r; ++i) {
        if (is_odd_power[i]) {
            consecutive_count++;
        } else {
            consecutive_count = 0; // Сбрасываем счетчик, если встречается не-нечётно-степенное число.
        }

        // Если мы нашли m подряд идущих нечётно-степенных чисел.
        if (consecutive_count == m) {
            first_in_sequence = i - m + 1; // Последовательность начинается с (текущее_число - m + 1).
            break; // Нам нужно наименьшее начальное число, поэтому прерываемся после нахождения первого.
        }
    }

    // Выводим результат.
    if (first_in_sequence != -1) {
        for (int i = 0; i < m; ++i) {
            std::cout << first_in_sequence + i << (i == m - 1 ? "" : " ");
        }
        std::cout << std::endl;
    } else {
        std::cout << -1 << std::endl;
    }

    return 0;
}