
#include <iostream>
#include <vector>
#include <algorithm>
#include <cmath>
#include <limits> // Для std::numeric_limits

// Используем быстрый ввод/вывод
void fast_io() {
    std::ios_base::sync_with_stdio(false);
    std::cin.tie(NULL);
    std::cout.tie(NULL);
}

// Структура для представления прямоугольника
struct Rect {
    long long x1, y1, x2, y2;
};

// Константы для вычислений с плавающей точкой
const double EPSILON = 1e-9; // Точность для сравнения чисел с плавающей точкой
const double MAX_COORD_VAL = 1e9; // Максимальное значение координаты
const double MAX_SLOPE_BOUND = 2e9; // Максимальное возможное значение наклона. 
                                    // Теоретически может быть ~10^9 (y-разница / x-разница), 
                                    // берем запас.
const int ITERATIONS = 100; // Количество итераций для тернарного поиска

// Функция для вычисления H(m) при m > 0
// H_plus(m) = min_i(y2_i - m*x1_i) - max_i(y1_i - m*x2_i)
double evaluate_h_plus(double m_val, const std::vector<Rect>& rectangles) {
    double current_max_L = -std::numeric_limits<double>::infinity(); // Максимальная нижняя граница для 'b'
    double current_min_U = std::numeric_limits<double>::infinity();  // Минимальная верхняя граница для 'b'

    for (const auto& rect : rectangles) {
        // Для m > 0, прямая y=mx+b возрастает.
        // Минимальное значение mx+b на интервале [x1,x2] равно m*x1+b. Максимальное mx+b равно m*x2+b.
        // Условия пересечения: m*x1+b <= y2_i И m*x2+b >= y1_i
        // Выражаем b:
        // b <= y2_i - m*x1_i (это U_i - верхняя граница b для данного прямоугольника)
        // b >= y1_i - m*x2_i (это L_i - нижняя граница b для данного прямоугольника)
        double L_i = (double)rect.y1 - m_val * rect.x2;
        double U_i = (double)rect.y2 - m_val * rect.x1;
        current_max_L = std::max(current_max_L, L_i);
        current_min_U = std::min(current_min_U, U_i);
    }
    return current_min_U - current_max_L;
}

// Функция для вычисления H(m) при m < 0
// H_minus(m) = min_i(y2_i - m*x2_i) - max_i(y1_i - m*x1_i)
double evaluate_h_minus(double m_val, const std::vector<Rect>& rectangles) {
    double current_max_L = -std::numeric_limits<double>::infinity();
    double current_min_U = std::numeric_limits<double>::infinity();

    for (const auto& rect : rectangles) {
        // Для m < 0, прямая y=mx+b убывает.
        // Минимальное значение mx+b на интервале [x1,x2] равно m*x2+b. Максимальное mx+b равно m*x1+b.
        // Условия пересечения: m*x2+b <= y2_i И m*x1+b >= y1_i
        // Выражаем b:
        // b <= y2_i - m*x2_i (это U_i)
        // b >= y1_i - m*x1_i (это L_i)
        double L_i = (double)rect.y1 - m_val * rect.x1;
        double U_i = (double)rect.y2 - m_val * rect.x2;
        current_max_L = std::max(current_max_L, L_i);
        current_min_U = std::min(current_min_U, U_i);
    }
    return current_min_U - current_max_L;
}


// Функция для решения одного набора тестовых данных
bool solve_test_case() {
    int n;
    std::cin >> n;
    std::vector<Rect> rectangles(n);
    for (int i = 0; i < n; ++i) {
        std::cin >> rectangles[i].x1 >> rectangles[i].y1 >> rectangles[i].x2 >> rectangles[i].y2;
    }

    // --- Проверка на горизонтальную линию (m=0) ---
    long long max_y1 = 0; // Минимально возможное y1 равно 0
    long long min_y2 = MAX_COORD_VAL + 1; // Максимально возможное y2 равно 10^9
    for (const auto& rect : rectangles) {
        max_y1 = std::max(max_y1, rect.y1);
        min_y2 = std::min(min_y2, rect.y2);
    }
    if (max_y1 <= min_y2) {
        return true;
    }

    // --- Проверка на вертикальную линию (m=бесконечность) ---
    long long max_x1 = 0; // Минимально возможное x1 равно 0
    long long min_x2 = MAX_COORD_VAL + 1; // Максимально возможное x2 равно 10^9
    for (const auto& rect : rectangles) {
        max_x1 = std::max(max_x1, rect.x1);
        min_x2 = std::min(min_x2, rect.x2);
    }
    if (max_x1 <= min_x2) {
        return true;
    }

    // --- Тернарный поиск для m > 0 ---
    double left_m_plus = EPSILON;
    double right_m_plus = MAX_SLOPE_BOUND;

    for (int i = 0; i < ITERATIONS; ++i) {
        double m1 = left_m_plus + (right_m_plus - left_m_plus) / 3.0;
        double m2 = right_m_plus - (right_m_plus - left_m_plus) / 3.0;
        if (evaluate_h_plus(m1, rectangles) < evaluate_h_plus(m2, rectangles)) {
            left_m_plus = m1;
        } else {
            right_m_plus = m2;
        }
    }
    // Проверяем максимально найденное значение H(m). Если оно неотрицательно (с учетом погрешности), то линия существует.
    if (evaluate_h_plus(left_m_plus, rectangles) >= -EPSILON) {
        return true;
    }

    // --- Тернарный поиск для m < 0 ---
    double left_m_minus = -MAX_SLOPE_BOUND;
    double right_m_minus = -EPSILON;

    for (int i = 0; i < ITERATIONS; ++i) {
        double m1 = left_m_minus + (right_m_minus - left_m_minus) / 3.0;
        double m2 = right_m_minus - (right_m_minus - left_m_minus) / 3.0;
        if (evaluate_h_minus(m1, rectangles) < evaluate_h_minus(m2, rectangles)) {
            left_m_minus = m1;
        } else {
            right_m_minus = m2;
        }
    }
    if (evaluate_h_minus(left_m_minus, rectangles) >= -EPSILON) {
        return true;
    }

    return false; // Ни одна из проверок не дала положительного результата
}

int main() {
    fast_io(); // Включаем быструю работу с вводом/выводом
    int T;
    std::cin >> T; // Читаем количество наборов тестовых данных
    while (T--) {
        if (solve_test_case()) {
            std::cout << "Yes\n";
        } else {
            std::cout << "No\n";
        }
    }
    return 0;
}

