
import sys

# Используем float для координат и вычислений. В Python float обычно соответствует double-precision (64-bit).
# sys.stdin.readline быстрее, чем input()
# Количество итераций для тернарного поиска для достаточной точности
ITERATIONS = 100
# Максимальное значение координаты (x2 или y2)
MAX_COORD_VAL = 10**9 
# Достаточно малое положительное число для начала/конца диапазона тернарного поиска
EPSILON = 1e-12 

# Максимальное значение для абсолютного значения наклона (slope).
# Например, линия, проходящая через (0,0) и (1, 10^9) имеет наклон 10^9.
# Используем немного больший диапазон для безопасности.
MAX_SLOPE_BOUND = 1e10 

def solve_test_case():
    n = int(sys.stdin.readline())
    rectangles = []
    for _ in range(n):
        x1, y1, x2, y2 = map(int, sys.stdin.readline().split())
        rectangles.append((x1, y1, x2, y2))

    # Проверка на наличие горизонтальной линии-трансверсали (m=0)
    # Прямая y = C пересекает все прямоугольники, если C находится в пересечении всех интервалов [y1_i, y2_i]
    max_y1 = -1 # Наименьшее возможное y1 равно 0, поэтому -1 — безопасное начальное значение
    min_y2 = MAX_COORD_VAL + 1 # Наибольшее возможное y2 равно 10^9, поэтому это безопасное начальное значение
    for x1, y1, x2, y2 in rectangles:
        max_y1 = max(max_y1, y1)
        min_y2 = min(min_y2, y2)
    if max_y1 <= min_y2:
        sys.stdout.write("Yes\n")
        return

    # Проверка на наличие вертикальной линии-трансверсали (m=бесконечность)
    # Прямая x = C пересекает все прямоугольники, если C находится в пересечении всех интервалов [x1_i, x2_i]
    max_x1 = -1 # Наименьшее возможное x1 равно 0, поэтому -1 — безопасное начальное значение
    min_x2 = MAX_COORD_VAL + 1 # Наибольшее возможное x2 равно 10^9, поэтому это безопасное начальное значение
    for x1, y1, x2, y2 in rectangles:
        max_x1 = max(max_x1, x1)
        min_x2 = min(min_x2, x2)
    if max_x1 <= min_x2:
        sys.stdout.write("Yes\n")
        return

    # Проверка на наличие наклонной линии-трансверсали (ни вертикальной, ни горизонтальной) с помощью тернарного поиска

    # Функция для вычисления H(m) для m > 0
    # H(m) = min_i(y2_i - m*x1_i) - max_i(y1_i - m*x2_i)
    # Это значение представляет максимально возможную "ширину" допустимого интервала для 'b' (y-пересечения) при заданном 'm' (наклоне).
    # Мы ищем 'm', для которого этот диапазон неотрицателен.
    def evaluate_h_plus(m_val):
        current_max_L = -float('inf') # Максимальная нижняя граница для 'b'
        current_min_U = float('inf')  # Минимальная верхняя граница для 'b'
        for x1, y1, x2, y2 in rectangles:
            # Для m > 0, прямая y=mx+b возрастает.
            # Минимальное значение mx+b на интервале [x1, x2] равно mx1+b.
            # Максимальное значение mx+b на интервале [x1, x2] равно mx2+b.
            # Условие: mx1+b <= y2_i И mx2+b >= y1_i
            # Перегруппируем для b:
            # b <= y2_i - m*x1_i (это U_i - верхняя граница b для данного прямоугольника)
            # b >= y1_i - m*x2_i (это L_i - нижняя граница b для данного прямоугольника)
            L_i = y1 - m_val * x2
            U_i = y2 - m_val * x1
            current_max_L = max(current_max_L, L_i)
            current_min_U = min(current_min_U, U_i)
        return current_min_U - current_max_L

    # Тернарный поиск для m в диапазоне (0, MAX_SLOPE_BOUND]
    left_m_plus = EPSILON
    right_m_plus = MAX_SLOPE_BOUND

    for _ in range(ITERATIONS):
        m1 = left_m_plus + (right_m_plus - left_m_plus) / 3
        m2 = right_m_plus - (right_m_plus - left_m_plus) / 3
        # Если H(m1) меньше H(m2), это означает, что максимум, скорее всего, находится правее
        if evaluate_h_plus(m1) < evaluate_h_plus(m2):
            left_m_plus = m1
        else: # Иначе максимум, скорее всего, находится левее или m1 ближе к максимуму
            right_m_plus = m2
    
    # После итераций left_m_plus (или right_m_plus) находится близко к 'm', которое максимизирует H_plus(m).
    # Проверяем, является ли это максимальное значение неотрицательным (с учетом небольшой погрешности float).
    if evaluate_h_plus(left_m_plus) >= -EPSILON:
        sys.stdout.write("Yes\n")
        return

    # Функция для вычисления H(m) для m < 0
    # H(m) = min_i(y2_i - m*x2_i) - max_i(y1_i - m*x1_i)
    def evaluate_h_minus(m_val):
        current_max_L = -float('inf')
        current_min_U = float('inf')
        for x1, y1, x2, y2 in rectangles:
            # Для m < 0, прямая y=mx+b убывает.
            # Минимальное значение mx+b на интервале [x1, x2] равно mx2+b.
            # Максимальное значение mx+b на интервале [x1, x2] равно mx1+b.
            # Условие: mx2+b <= y2_i И mx1+b >= y1_i
            # Перегруппируем для b:
            # b <= y2_i - m*x2_i (это U_i)
            # b >= y1_i - m*x1_i (это L_i)
            L_i = y1 - m_val * x1
            U_i = y2 - m_val * x2
            current_max_L = max(current_max_L, L_i)
            current_min_U = min(current_min_U, U_i)
        return current_min_U - current_max_L

    # Тернарный поиск для m в диапазоне [-MAX_SLOPE_BOUND, -EPSILON)
    left_m_minus = -MAX_SLOPE_BOUND
    right_m_minus = -EPSILON

    for _ in range(ITERATIONS):
        m1 = left_m_minus + (right_m_minus - left_m_minus) / 3
        m2 = right_m_minus - (right_m_minus - left_m_minus) / 3
        if evaluate_h_minus(m1) < evaluate_h_minus(m2):
            left_m_minus = m1
        else:
            right_m_minus = m2

    if evaluate_h_minus(left_m_minus) >= -EPSILON:
        sys.stdout.write("Yes\n")
        return

    # Если ни одна из проверок не дала положительного результата
    sys.stdout.write("No\n")

# Чтение количества наборов тестовых данных
num_test_cases = int(sys.stdin.readline())
for _ in range(num_test_cases):
    solve_test_case()

