import sys
from functools import lru_cache

# Увеличиваем лимит рекурсии, так как глубина рекурсии может достигать ~10^5 (длина R в битах)
sys.setrecursionlimit(150000)

MOD = 10**9 + 7

# Глобальные переменные для удобства доступа из рекурсивной функции с lru_cache
K_val = 0
R_bin = [] # Список битов числа R (MSB сначала)

@lru_cache(None) # Декоратор для мемоизации результатов
def solve(idx, rem, tight, started):
    """
    Функция цифрового динамического программирования.

    :param idx: Текущий индекс бита, который мы рассматриваем (от 0 до len(R_bin)-1).
    :param rem: Текущий popcount по модулю K.
    :param tight: True, если мы ограничены битами R, False иначе.
    :param started: True, если мы уже поставили хотя бы одну '1' (не обрабатываем ведущие нули).
    :return: Количество чисел, удовлетворяющих условиям, для текущего префикса.
    """
    # Базовый случай: если все биты обработаны
    if idx == len(R_bin):
        # Если popcount по модулю K равно 0, это допустимое число
        return 1 if rem == 0 else 0

    ans = 0
    # Определяем верхнюю границу для текущего бита (0 или 1)
    # Если tight, мы можем поставить только бит <= R_bin[idx]. Иначе можно поставить 0 или 1.
    upper_bound = R_bin[idx] if tight else 1

    for digit in range(upper_bound + 1):
        new_tight = tight and (digit == upper_bound)
        new_started = started or (digit == 1)

        if not new_started: # Мы все еще обрабатываем ведущие нули
            # Если ставим '0' и еще не начали число, popcount остается 0, rem не меняется.
            ans = (ans + solve(idx + 1, rem, new_tight, new_started)) % MOD
        else: # Число уже началось (первая '1' была размещена)
            # Добавляем 'digit' к popcount и берем по модулю K.
            ans = (ans + solve(idx + 1, (rem + digit) % K_val, new_tight, new_started)) % MOD
    
    return ans

def main():
    global K_val, R_bin # Используем глобальные переменные

    # Чтение входных данных
    _, K_val_str = sys.stdin.readline().split()
    K_val = int(K_val_str)
    R_str = sys.stdin.readline().strip()

    # Краевой случай: R = "0"
    # popcount(0) = 0. 0 % K = 0 для любого K >= 1.
    # Таким образом, число 0 удовлетворяет условию. Ответ 1.
    if R_str == "0":
        print(1)
        return

    # Преобразуем десятичное R в двоичное R_bin, используя встроенные возможности Python для больших чисел
    R_val = int(R_str)
    R_bin_str = bin(R_val)[2:] # Удаляем префикс "0b"
    R_bin = [int(bit) for bit in R_bin_str] # Список битов (MSB сначала)

    # Оптимизация: Если K > максимальное возможное popcount
    # Максимальное popcount для числа x <= R равно длине двоичного представления R.
    max_popcount_val = len(R_bin)
    if K_val > max_popcount_val:
        # Если K больше любого возможного popcount, то popcount(x) % K == 0 означает popcount(x) == 0.
        # Единственное число с popcount(x) = 0 - это x = 0.
        # Таким образом, только число 0 удовлетворяет условиям. Ответ 1.
        print(1)
        return

    # Запускаем цифровую DP.
    # Начинаем с самого старшего бита (индекс 0), начальный остаток popcount = 0,
    # tight=True (ограничены R), started=False (еще не начали ставить единицы).
    result = solve(0, 0, True, False)

    print(result)

if __name__ == "__main__":
    main()