from math import isclose

def point_on_segment(px, py, x1, y1, x2, y2):
    """Проверяет, лежит ли точка (px,py) на отрезке (x1,y1)-(x2,y2)."""
    if not (min(x1, x2) <= px <= max(x1, x2) and min(y1, y2) <= py <= max(y1, y2)):
        return False
    # Векторное произведение (должно быть 0 для коллинеарных)
    cross = (px - x1) * (y2 - y1) - (py - y1) * (x2 - x1)
    return isclose(cross, 0.0, abs_tol=1e-9)

def line_intersects_rect(A, B, rect):
    """
    Проверяет, пересекает ли отрезок AB границу прямоугольника rect.
    rect = (x1, y1, x2, y2)
    """
    x1, y1, x2, y2 = rect
    # Проверяем пересечение с каждым ребром
    edges = [
        (x1, y1, x2, y1),  # нижнее
        (x2, y1, x2, y2),  # правое
        (x2, y2, x1, y2),  # верхнее
        (x1, y2, x1, y1)   # левое
    ]
    for ex1, ey1, ex2, ey2 in edges:
        # Проверка пересечения отрезков AB и (ex1,ey1)-(ex2,ey2)
        dx1 = B[0] - A[0]
        dy1 = B[1] - A[1]
        dx2 = ex2 - ex1
        dy2 = ey2 - ey1

        det = dx1 * dy2 - dy1 * dx2
        if det == 0:
            # Параллельны — проверяем, лежит ли хотя бы одна точка на ребре
            if point_on_segment(A[0], A[1], ex1, ey1, ex2, ey2) or \
               point_on_segment(B[0], B[1], ex1, ey1, ex2, ey2):
                return True
            continue
        t = ((ex1 - A[0]) * dy2 - (ey1 - A[1]) * dx2) / det
        u = ((ex1 - A[0]) * dy1 - (ey1 - A[1]) * dx1) / det
        if 0 <= t <= 1 and 0 <= u <= 1:
            return True
    return False

def rect_touched_by_line(line_point1, line_point2, rect):
    """
    Проверяет, касается ли прямая (через line_point1 и line_point2) прямоугольника rect.
    Прямая бесконечна, поэтому проверяем пересечение луча с границей rect.
    """
    A, B = line_point1, line_point2
    x1, y1, x2, y2 = rect


    # Если прямая проходит через угол
    for px, py in [(x1,y1), (x2,y1), (x2,y2), (x1,y2)]:
        if point_on_segment(px, py, A[0], A[1], B[0], B[1]):
            return True

    # Если пересекает хотя бы одно ребро
    return line_intersects_rect(A, B, rect)


def get_candidate_lines(rect1, rect2):
    """Генерирует прямые-кандидаты через углы rect1 и rect2."""
    lines = []
    x11, y11, x12, y12 = rect1
    x21, y21, x22, y22 = rect2
    corners1 = [(x11,y11), (x12,y11), (x12,y12), (x11,y12)]
    corners2 = [(x21,y21), (x22,y21), (x22,y22), (x21,y22)]
    # Прямые через пары углов
    for p1 in corners1:
        for p2 in corners2:
            lines.append((p1, p2))


    # Горизонтальные и вертикальные прямые через рёбра
    for y in [y11, y12]:
        lines.append(((0, y), (1, y)))  # y = const
    for x in [x11, x12]:
        lines.append(((x, 0), (x, 1)))  # x = const


    for y in [y21, y22]:
        lines.append(((0, y), (1, y)))
    for x in [x21, x22]:
        lines.append(((x, 0), (x, 1)))


    return lines

def solve_case(n, rects):
    if n <= 2:
        return "Yes"

    rect1 = rects[0]
    rect2 = rects[1]
    candidates = get_candidate_lines(rect1, rect2)

    for A, B in candidates:
        touches_all = True
        for rect in rects:
            if not rect_touched_by_line(A, B, rect):
                touches_all = False
                break
        if touches_all:
            return "Yes"
    return "No"

def main():
    T = int(input())
    for _ in range(T):
        n = int(input())
        rects = []
        for _ in range(n):
            x1, y1, x2, y2 = map(int, input().split())
            rects.append((x1, y1, x2, y2))
        result = solve_case(n, rects)
        print(result)

if __name__ == "__main__":
    main()

input()
