import sys

def solve():
    """
    Рассчитывает количество клеток каждого цвета в n x m сетке
    с диагональной раскраской по k цветам.
    """
    try:
        # Чтение входных данных из стандартного ввода (sys.stdin)
        line = sys.stdin.read().strip().split()
        if not line:
            return

        n = int(line[0])
        m = int(line[1])
        k = int(line[2])

    except (IndexError, ValueError):
        # Обработка ошибок ввода, если данные неполные или некорректны
        return

    # Убеждаемся, что n <= m для упрощения логики трех областей
    if n > m:
        n, m = m, n

    # Массив для хранения итогового количества клеток каждого из k цветов (индексация с 0)
    colors_count = [0] * k
    
    # 1. Первая область: Диагонали D от 1 до n. Длина диагонали = D.
    # Количество полных циклов k цветов в этой области
    full_cycles = (n - 1) // k
    remaining = (n - 1) % k
    
    # Сумма длин диагоналей для каждого цвета в полных циклах
    # Для цвета c (0 до k-1) сумма длин диагоналей: (c+1) + (c+1+k) + ...
    # Это арифметическая прогрессия.
    for c in range(k):
        count_of_diagonals_with_color_c = full_cycles
        # Если цвет c попадает в оставшиеся диагонали, увеличиваем их количество
        if c < remaining:
            count_of_diagonals_with_color_c += 1
        
        # Общая длина (количество клеток) для цвета c в этой области:
        # Formula for sum of AP: count * (2*a1 + (count-1)*d) / 2
        # a1 = c + 1
        # d = k
        # count = count_of_diagonals_with_color_c
        
        if count_of_diagonals_with_color_c > 0:
            a1 = c + 1
            last_term = a1 + (count_of_diagonals_with_color_c - 1) * k
            colors_count[c] += count_of_diagonals_with_color_c * (a1 + last_term) // 2

    # 2. Вторая область: Диагонали D от n до m. Длина диагонали = n.
    # Диагональ n имеет цвет (n-1) % k.
    # Количество диагоналей в этой области: m - n + 1
    num_middle_diagonals = m - n + 1
    
    # Начальный цвет для этой области
    start_color_index = (n - 1) % k
    
    for i in range(num_middle_diagonals):
        current_color_index = (start_color_index + i) % k
        colors_count[current_color_index] += n # Длина всех диагоналей здесь равна n

    # 3. Третья область: Диагонали D от m+1 до n+m-1. Длина диагонали уменьшается от n-1 до 1.
    # По сути, это зеркальное отражение первой области, но с другим смещением цветов.
    # Используем ту же логику, что и для первой области, но для длин от n-1 до 1.
    
    # Количество диагоналей в третьей области: n-1
    num_last_section = n - 1
    full_cycles = num_last_section // k
    remaining = num_last_section % k

    # Начальный цвет для этой области: (m) % k.
    # Диагонали имеют длины: n-1, n-2, ..., 1. 
    # Удобнее считать их длины как 1, 2, ..., n-1, но в обратном порядке цветов.

    # Или можно просто использовать симметрию: количество клеток в этой области
    # для каждого цвета будет почти таким же, как в первой области, 
    # но нужно учесть смещение стартового цвета (m % k).

    start_color_index_third = (m % k)
    
    for c in range(k):
        # Соответствующий "зеркальный" цвет из первой области
        mirrored_color = (start_color_index_third + c) % k
        
        count_of_diagonals_with_color_c = full_cycles
        if c < remaining:
            count_of_diagonals_with_color_c += 1
        
        if count_of_diagonals_with_color_c > 0:
            # Длины здесь: 1, 2, ..., n-1 (при правильном порядке цветов)
            a1 = c + 1
            last_term = a1 + (count_of_diagonals_with_color_c - 1) * k
            sum_of_lengths = count_of_diagonals_with_color_c * (a1 + last_term) // 2
            
            colors_count[mirrored_color] += sum_of_lengths

    # Вывод результатов
    print(*(colors_count))

if __name__ == "__main__":
    solve()
