import sys

# Define the maximum value for r.
# All precomputations will be done up to this limit.
MAX_R = 10**6

# min_prime_factor[i] will store the smallest prime factor of i.
# Initialized to 0, will be filled by the sieve.
min_prime_factor = [0] * (MAX_R + 1)

# is_odd_power_arr[i] will be True if i is an odd-power number, False otherwise.
# Initialized to False.
is_odd_power_arr = [False] * (MAX_R + 1)

def sieve_min_prime_factor():
    """
    Implements a sieve to find the smallest prime factor for every number up to MAX_R.
    This is a standard optimization for prime factorization.
    """
    # Initialize each number's smallest prime factor to itself.
    # This means numbers that remain `i` after the sieve are prime.
    for i in range(2, MAX_R + 1):
        min_prime_factor[i] = i
    
    # Iterate through potential prime factors.
    # We only need to check primes up to sqrt(MAX_R) because any composite number N
    # must have at least one prime factor less than or equal to sqrt(N).
    for p in range(2, int(MAX_R**0.5) + 1):
        # If min_prime_factor[p] is still p, then p is a prime number.
        if min_prime_factor[p] == p:
            # Mark multiples of p.
            # We start from p*p because smaller multiples (like 2p, 3p, etc.)
            # would have already been marked by their smaller prime factors (2, 3, etc.).
            for multiple in range(p * p, MAX_R + 1, p):
                # Only update min_prime_factor if it hasn't been set by a smaller prime.
                # This ensures min_prime_factor[multiple] always stores the *smallest* prime factor.
                if min_prime_factor[multiple] == multiple:
                    min_prime_factor[multiple] = p

def precompute_odd_power_numbers():
    """
    Determines for each number from 2 to MAX_R if it is an odd-power number.
    This uses the precomputed smallest prime factors for efficient factorization.
    """
    # The problem states l >= 2, so we don't need to consider 1.
    # If 1 were considered, it would be an odd-power number (vacuously true).
    
    for i in range(2, MAX_R + 1):
        temp_n = i
        is_current_odd_power = True
        
        # Factorize temp_n using the smallest prime factor array
        while temp_n > 1:
            p = min_prime_factor[temp_n] # Get the smallest prime factor of temp_n
            count = 0
            
            # Count how many times p divides temp_n (this is the exponent of p)
            while temp_n % p == 0:
                temp_n //= p # Divide temp_n by p
                count += 1   # Increment exponent count
            
            # If the exponent of any prime factor is even, the number is not odd-power
            if count % 2 == 0:
                is_current_odd_power = False
                break # No need to check further prime factors for this number
        
        is_odd_power_arr[i] = is_current_odd_power

def solve():
    """
    Main function to solve the problem:
    1. Precomputes necessary data.
    2. Reads input m, l, r.
    3. Finds the first sequence of m consecutive odd-power numbers in [l, r].
    4. Prints the result or -1 if no such sequence exists.
    """
    # Step 1: Precompute the smallest prime factors and odd-power status for all numbers.
    sieve_min_prime_factor()
    precompute_odd_power_numbers()

    # Step 2: Read input m, l, r from standard input.
    m = int(sys.stdin.readline())
    l, r = map(int, sys.stdin.readline().split())

    current_consecutive_odd_powers = 0
    start_of_sequence = -1 # Stores the starting number of the found sequence

    # Step 3: Iterate through the range [l, r] using a sliding window approach.
    # We count consecutive odd-power numbers. If a non-odd-power number is encountered,
    # the count resets. If the count reaches 'm', we found our sequence.
    for i in range(l, r + 1):
        if is_odd_power_arr[i]:
            current_consecutive_odd_powers += 1
            # If we have found 'm' consecutive odd-power numbers,
            # this 'i' is the end of the sequence.
            if current_consecutive_odd_powers == m:
                start_of_sequence = i - m + 1 # Calculate the start of the sequence
                break # We found the first such sequence, so we can stop searching.
        else:
            # If 'i' is not an odd-power number, break the current consecutive streak.
            current_consecutive_odd_powers = 0

    # Step 4: Print the result.
    if start_of_sequence != -1:
        # Generate the sequence of numbers and format for output.
        output_numbers = [str(x) for x in range(start_of_sequence, start_of_sequence + m)]
        sys.stdout.write(" ".join(output_numbers) + "\n")
    else:
        # If no sequence was found, print -1.
        sys.stdout.write("-1\n")

# Call the main solution function.
solve()
