import sys

def solve_inequality(segments_A, segments_B):
    # segments_A: list of (x1, y1) for lines f_i(k) = x1 * k + y1
    # segments_B: list of (x2, y2) for lines g_j(k) = x2 * k + y2
    # We want: max_i f_i(k) <= min_j g_j(k) for some k >= 0
    # Equivalently: F(k) = max_i f_i(k) - min_j g_j(k) <= 0
    # Find min of F(k) for k >= 0
    
    # Step 1: find upper envelope of segments_A and lower envelope of segments_B
    # Since we only need to check if min F(k) <= 0, we can check at:
    # k = 0, k = inf, and intersections of lines from A with lines from B
    
    nA = len(segments_A)
    nB = len(segments_B)
    
    # Helper: value at k
    def F(k):
        maxA = max(x*k + y for x, y in segments_A)
        minB = min(x*k + y for x, y in segments_B)
        return maxA - minB
    
    # Check k = 0
    if F(0.0) <= 0:
        return True
    
    # Check k large, say 1e9
    if F(1e9) <= 0:
        return True
    
    # Check intersections between each pair (i from A, j from B)
    # Solve x1_i * k + y1_i = x2_j * k + y2_j
    # k = (y2_j - y1_i) / (x1_i - x2_j) if denominator not zero
    for x1, y1 in segments_A:
        for x2, y2 in segments_B:
            denom = x1 - x2
            if denom == 0:
                continue
            k = (y2 - y1) / denom
            if k >= -1e-12:  # allow small negative due to precision
                if F(max(k, 0.0)) <= 1e-9:
                    return True
    return False

def solve_one(rects):
    n = len(rects)
    
    # Special cases: vertical and horizontal lines
    if max(r[0] for r in rects) <= min(r[2] for r in rects):
        return "Yes"
    if max(r[1] for r in rects) <= min(r[3] for r in rects):
        return "Yes"
    
    # Case 1: a >= 0, b >= 0
    # L = a*x1 + b*y1, U = a*x2 + b*y2
    # Divide by b > 0: k = a/b >= 0
    # max(k*x1 + y1) <= min(k*x2 + y2)
    segs_A = [(r[0], r[1]) for r in rects]  # x1, y1
    segs_B = [(r[2], r[3]) for r in rects]  # x2, y2
    if solve_inequality(segs_A, segs_B):
        return "Yes"
    
    # Case 2: a >= 0, b <= 0
    # L = a*x1 - |b|*y2, U = a*x2 - |b|*y1
    # Divide by -b > 0: k = a/(-b) >= 0
    # max(k*x1 + y2) <= min(k*x2 + y1)
    segs_A = [(r[0], r[3]) for r in rects]  # x1, y2
    segs_B = [(r[2], r[1]) for r in rects]  # x2, y1
    if solve_inequality(segs_A, segs_B):
        return "Yes"
    
    # Case 3: a <= 0, b >= 0
    # L = -|a|*x2 + b*y1, U = -|a|*x1 + b*y2
    # Divide by b > 0: k = |a|/b >= 0
    # max(-k*x2 + y1) <= min(-k*x1 + y2)
    segs_A = [(-r[2], r[1]) for r in rects]  # -x2, y1
    segs_B = [(-r[0], r[3]) for r in rects]  # -x1, y2
    if solve_inequality(segs_A, segs_B):
        return "Yes"
    
    # Case 4: a <= 0, b <= 0
    # L = -|a|*x2 - |b|*y2, U = -|a|*x1 - |b|*y1
    # Divide by -b > 0: k = |a|/|b| >= 0
    # max(-k*x2 + y2) <= min(-k*x1 + y1)
    segs_A = [(-r[2], r[3]) for r in rects]  # -x2, y2
    segs_B = [(-r[0], r[1]) for r in rects]  # -x1, y1
    if solve_inequality(segs_A, segs_B):
        return "Yes"
    
    return "No"

def main():
    import sys
    input = sys.stdin.read
    data = input().split()
    t = int(data[0])
    idx = 1
    out = []
    for _ in range(t):
        n = int(data[idx]); idx += 1
        rects = []
        for __ in range(n):
            x1 = int(data[idx]); y1 = int(data[idx+1])
            x2 = int(data[idx+2]); y2 = int(data[idx+3])
            idx += 4
            rects.append((x1, y1, x2, y2))
        out.append(solve_one(rects))
    sys.stdout.write("\n".join(out))

if __name__ == "__main__":
    main()
