import sys
import math

EPS = 1e-12
INF = 1e100

def build_envelope(lines, mode):
    if not lines:
        return []
    lines.sort(key=lambda t: (t[0], t[1]))
    filtered = []
    if mode == 'max':
        for m,b in lines:
            if filtered and abs(m - filtered[-1][0]) < EPS:
                if b > filtered[-1][1]:
                    filtered[-1] = (m,b)
            else:
                filtered.append((m,b))
    else:
        for m,b in lines:
            if filtered and abs(m - filtered[-1][0]) < EPS:
                if b < filtered[-1][1]:
                    filtered[-1] = (m,b)
            else:
                filtered.append((m,b))

    hull = []
    def intersect_x(m1,b1,m2,b2):
        if abs(m1 - m2) < EPS:
            return INF if abs(b1 - b2) < EPS else (INF if (mode=='max' and b1>b2) else -INF)
        return (b2 - b1) / (m1 - m2)

    for m,b in filtered:
        if not hull:
            hull.append((m,b,-INF))
            continue
        while True:
            m0,b0,x0 = hull[-1]
            x = intersect_x(m0,b0,m,b)
            if x <= x0 + EPS:
                hull.pop()
                if not hull:
                    break
                else:
                    continue
            else:
                hull.append((m,b,x))
                break
        if not hull:
            hull.append((m,b,-INF))
    return hull

def exists_r_with_overlap(upper, lower):
    i = 0
    j = 0
    ni = len(upper)
    nj = len(lower)
    while i < ni and j < nj:
        mi,bi,si = upper[i]
        mj,bj,sj = lower[j]
        next_i = upper[i+1][2] if i+1<ni else INF
        next_j = lower[j+1][2] if j+1<nj else INF
        seg_l = max(si, sj, 0.0)
        seg_r = min(next_i, next_j)
        if seg_r + EPS >= seg_l:
            for r in (seg_l, seg_r if seg_r < INF/2 else None, (seg_l + min(seg_r, seg_l+1.0))/2):
                if r is None:
                    continue
                valU = mi * r + bi
                valV = mj * r + bj
                if valU <= valV + 1e-9:
                    return True
        if next_i < next_j:
            i += 1
        else:
            j += 1
    return False

def solve_one(rects):
    n = len(rects)
    max_x1 = max(r[0] for r in rects)
    min_x2 = min(r[2] for r in rects)
    if max_x1 <= min_x2 + EPS:
        return True
    max_y1 = max(r[1] for r in rects)
    min_y2 = min(r[3] for r in rects)
    if max_y1 <= min_y2 + EPS:
        return True

    signs = [(1,1),(1,-1),(-1,1),(-1,-1)]
    for sx, sy in signs:
        L_lines = []
        R_lines = []
        valid = True
        for x1,y1,x2,y2 in rects:
            tx1 = sx * x1
            tx2 = sx * x2
            ty1 = sy * y1
            ty2 = sy * y2
            xl = min(tx1, tx2)
            xr = max(tx1, tx2)
            yl = min(ty1, ty2)
            yr = max(ty1, ty2)
            L_lines.append((xl, yl))
            R_lines.append((xr, yr))

        upper = build_envelope(L_lines, 'max')
        lower = build_envelope(R_lines, 'min')
        if not upper or not lower:
            continue
        if exists_r_with_overlap(upper, lower):
            return True
    return False

def main():
    data = sys.stdin.read().strip().split()
    if not data:
        return
    it = iter(data)
    T = int(next(it))
    out = []
    for _ in range(T):
        n = int(next(it))
        rects = []
        for i in range(n):
            x1 = float(next(it)); y1 = float(next(it)); x2 = float(next(it)); y2 = float(next(it))
            rects.append((x1,y1,x2,y2))
        ans = solve_one(rects)
        out.append("Yes" if ans else "No")
    print("\n".join(out))

if __name__ == "__main__":
    main()
