import sys

def can_hit_all_rectangles(rects):
    """Проверяем, существует ли прямая, касающаяся всех прямоугольников."""
    n = len(rects)
    
    # 1. Проверим простые случаи: горизонтальные и вертикальные прямые
    # Для горизонтальной прямой: проверяем, существует ли y такой, что он попадает во все [y1, y2]
    y_min = max(rect[1] for rect in rects)  # максимальный из нижних границ
    y_max = min(rect[3] for rect in rects)  # минимальный из верхних границ
    if y_min <= y_max:
        return True
    
    # Для вертикальной прямой: проверяем, существует ли x такой, что он попадает во все [x1, x2]
    x_min = max(rect[0] for rect in rects)  # максимальный из левых границ
    x_max = min(rect[2] for rect in rects)  # минимальный из правых границ
    if x_min <= x_max:
        return True
    
    # 2. Проверим направления 45° и 135°
    # Для направления 45°: проектируем на ось u = x - y
    # Для каждого прямоугольника: u_min = x1 - y2, u_max = x2 - y1
    u_min = max(rect[0] - rect[3] for rect in rects)  # x1 - y2
    u_max = min(rect[2] - rect[1] for rect in rects)  # x2 - y1
    if u_min <= u_max:
        return True
    
    # Для направления 135°: проектируем на ось v = x + y
    # Для каждого прямоугольника: v_min = x1 + y1, v_max = x2 + y2
    v_min = max(rect[0] + rect[1] for rect in rects)
    v_max = min(rect[2] + rect[3] for rect in rects)
    if v_min <= v_max:
        return True
    
    # 3. Более сложный случай: ищем направление через пары "критических" прямоугольников
    # Находим "крайние" прямоугольники по разным критериям
    
    # Ищем прямоугольники с минимальной и максимальной y-координатой
    min_y_rect = min(rects, key=lambda r: r[1])
    max_y_rect = max(rects, key=lambda r: r[3])
    
    # Ищем прямоугольники с минимальной и максимальной x-координатой
    min_x_rect = min(rects, key=lambda r: r[0])
    max_x_rect = max(rects, key=lambda r: r[2])
    
    # Проверяем направления через углы этих прямоугольников
    critical_rects = [min_y_rect, max_y_rect, min_x_rect, max_x_rect]
    
    # Добавим ещё несколько "крайних" прямоугольников
    # По центру масс (усреднённые координаты)
    avg_x = sum((r[0] + r[2]) / 2 for r in rects) / n
    avg_y = sum((r[1] + r[3]) / 2 for r in rects) / n
    
    # Находим прямоугольники ближайшие и дальние от центра
    def dist_to_center(rect):
        cx = (rect[0] + rect[2]) / 2
        cy = (rect[1] + rect[3]) / 2
        return (cx - avg_x)**2 + (cy - avg_y)**2
    
    closest = min(rects, key=dist_to_center)
    farthest = max(rects, key=dist_to_center)
    critical_rects.extend([closest, farthest])
    
    # Уникальные прямоугольники
    critical_rects = list({tuple(r) for r in critical_rects})
    
    # Для каждой пары критических прямоугольников проверяем направления через их углы
    for i in range(len(critical_rects)):
        for j in range(i + 1, len(critical_rects)):
            r1 = critical_rects[i]
            r2 = critical_rects[j]
            
            # Проверяем все пары углов
            for corner1 in [(r1[0], r1[1]), (r1[0], r1[3]), (r1[2], r1[1]), (r1[2], r1[3])]:
                for corner2 in [(r2[0], r2[1]), (r2[0], r2[3]), (r2[2], r2[1]), (r2[2], r2[3])]:
                    # Не проверяем одинаковые точки
                    if corner1 == corner2:
                        continue
                    
                    # Угол прямой через эти две точки
                    dx = corner2[0] - corner1[0]
                    dy = corner2[1] - corner1[1]
                    
                    if dx == 0 and dy == 0:
                        continue
                    
                    # Проверяем, касается ли прямая через эти две точки всех прямоугольников
                    # Для этого проектируем на перпендикулярное направление
                    # Перпендикулярное направление: (-dy, dx)
                    if dx == 0:  # вертикальная прямая - уже проверяли
                        continue
                    if dy == 0:  # горизонтальная прямая - уже проверяли
                        continue
                    
                    # Нормализуем вектор нормали
                    # Проверяем проекции всех прямоугольников
                    ok = True
                    
                    # Для эффективности используем накопление min/max
                    proj_min = float('inf')
                    proj_max = float('-inf')
                    
                    for rect in rects:
                        # Для прямоугольника находим min и max проекции его вершин
                        x1, y1, x2, y2 = rect
                        corners = [(x1, y1), (x1, y2), (x2, y1), (x2, y2)]
                        
                        # Проекция точки (x, y) на нормаль (-dy, dx)
                        projections = [-dy * x + dx * y for x, y in corners]
                        rect_min = min(projections)
                        rect_max = max(projections)
                        
                        # Обновляем общие границы
                        proj_min = min(proj_min, rect_max)  # Общая часть: min из максимумов
                        proj_max = max(proj_max, rect_min)  # Общая часть: max из минимумов
                    
                    # Если общая часть не пуста
                    if proj_max <= proj_min:
                        return True
    
    return False

def main():
    input_data = sys.stdin.read().strip().split()
    if not input_data:
        return
    
    t = int(input_data[0])
    idx = 1
    results = []
    
    for _ in range(t):
        n = int(input_data[idx])
        idx += 1
        rects = []
        for __ in range(n):
            x1 = int(input_data[idx])
            y1 = int(input_data[idx + 1])
            x2 = int(input_data[idx + 2])
            y2 = int(input_data[idx + 3])
            idx += 4
            rects.append((x1, y1, x2, y2))
        
        if can_hit_all_rectangles(rects):
            results.append("Yes")
        else:
            results.append("No")
    
    sys.stdout.write("\n".join(results))

if __name__ == "__main__":
    main()