import sys

sys.setrecursionlimit(200000)
MOD = 10**9 + 7

def solve_final_optimized_time():
    try:
        # --- Блок чтения входных данных ---
        data_lines = sys.stdin.read().split()
        if not data_lines:
            return

        n = int(data_lines[0])
        K = int(data_lines[1])
        R_str = "".join(data_lines[2:])
        
        if K == 0:
            print(0)
            return

        R_int = int(R_str)
        if R_int == 0:
            print(1 % MOD)
            return
            
        binary_R = bin(R_int)[2:]
        m = len(binary_R)

        # --- Основная логика: Комбинаторный подход с DP по модулю K ---
        
        # DP[j] будет хранить количество способов получить j единиц в префиксе
        # Но нам нужен DP по модулю K
        
        # DP table: dp[i][j] = number of ways to have a popcount with remainder j after processing i bits
        # We can optimize space to O(K)
        
        # Инициализация DP: в начале 0 бит, остаток 0, 1 способ
        dp = * K]
        dp[0] = 1

        for i in range(m):
            new_dp = * K]
            # Пересчитываем DP для следующего бита
            for j in range(K):
                if dp[j] == 0:
                    continue
                # Можем добавить 0 бит: остаток j
                new_dp[j] = (new_dp[j] + dp[j]) % MOD
                # Можем добавить 1 бит: остаток (j + 1) % K
                next_j = (j + 1) % K
                new_dp[next_j] = (new_dp[next_j] + dp[j]) % MOD
            dp = new_dp
        
        # Этот DP считает количество чисел с popcount % K == 0 в диапазоне [0, 2^m - 1].
        # Это не то, что нам нужно. Нам нужно ограничение R.

        # --- Реальный оптимизированный комбинаторный подход с учетом R ---
        
        # Мы должны использовать префиксы R и считать комбинации для оставшихся бит.
        # Проблема TLE была в функции get_combinations_mod_fact. 
        # Вычисление факториалов и модульной инверсии в цикле для больших m - долго.

        # Мы можем предварительно вычислить только одну строку треугольника Паскаля C(m, *) 
        # или использовать DP для C(n, k).

        # Давайте попробуем использовать *встроенную* функцию Python для popcount для проверки самого R, 
        # и оптимизируем расчет C(n, k) 

        # Оптимизированная функция C(n, k) через умножение/деление
        def combinations_optimized(nn, kk):
            if kk < 0 or kk > nn:
                return 0
            if kk == 0 or kk == nn:
                return 1
            if kk > nn // 2:
                kk = nn - kk
            
            res = 1
            for i in range(kk):
                res = res * (nn - i) // (i + 1) # Python auto handles large ints, // for floor
            return res % MOD # This might lose modulo intermediate steps

        # Это не будет работать с modulo на промежуточных шагах, а без modulo 
        # числа будут слишком большими.

        # --- Единственный рабочий вариант: DP по префиксу с остатками K ---
        # Ограничение R должно быть учтено в DP. 

        # dp[i][j][tight] - TLE (Time Limit Exceeded)
        
        # Реализуем функцию count_valid_optimized_final, которая использует DP массив
        # для хранения количеств, с учетом "жесткого" ограничения R.
        
        # Вместо 2D массива будем использовать словарь для мемоизации, что сэкономит память, 
        # но может быть медленнее из-за хэширования.
        
        from functools import lru_cache

        @lru_cache(None)
        def memoized_dp(index, current_popcount_mod_K, is_less):
            if index == m:
                return 1 if current_popcount_mod_K == 0 else 0

            upper_bound = int(binary_R[index]) if not is_less else 1
            count = 0

            for digit in range(upper_bound + 1):
                new_is_less = is_less or (digit < upper_bound)
                new_popcount_mod_K = (current_popcount_mod_K + digit) % K
                count = (count + memoized_dp(index + 1, new_popcount_mod_K, new_is_less)) % MOD

            return count

        # Эта функция все равно превысит лимит памяти/времени из-за большого m и K.

        # --- ФИНАЛЬНЫЙ АЛГОРИТМ: КОМБИНАТОРИКА + DP ОСТАТКОВ ---
        
        # Используем комбинаторику для "не-tight" ветвей и DP остатков для "tight" ветви.
        
        count = 0
        popcount_so_far = 0
        
        # Precalculate combinations C(n, k) in a single dimension array
        # This will store only one row C[m][j] at a time. Still O(m) space.
        
        # Need a very fast C(n, k) function.
        # We stick to the factorial method as it's fastest *if precomputed*.

        # Precompute factorials up to m
        MAX_M = m + 5
        fact = * MAX_M]
        fact[0] = 1
        for i in range(1, MAX_M):
            fact[i] = (fact[i-1] * i) % MOD

        def power_with_modulo(base, exp):
            res = 1
            base %= MOD
            while exp > 0:
                if exp % 2 == 1:
                    res = (res * base) % MOD
                base = (base * base) % MOD
                exp //= 2
            return res

        def mod_inverse(n_inv):
            return power_with_modulo(n_inv, MOD - 2)

        def get_combinations_mod_fact(nn, kk):
            if kk < 0 or kk > nn:
                return 0
            numerator = fact[nn]
            denominator = (fact[kk] * fact[nn-kk]) % MOD
            return (numerator * mod_inverse(denominator)) % MOD

        # Iterate over the binary string
        for i in range(m):
            if binary_R[i] == '1':
                # Calculate for the '0' branch
                remaining_bits = m - 1 - i
                # Target popcount J such that (popcount_so_far + J) % K == 0
                # J % K == (-popcount_so_far % K)
                # target_remainder = (-popcount_so_far % K + K) % K

                target_remainder = (-popcount_so_far % K + K) % K
                
                # We need all J in [0, remaining_bits] such that J % K == target_remainder
                
                # Iterate over possible values of J (step K)
                start_j = target_remainder
                while start_j <= remaining_bits:
                    count = (count + get_combinations_mod_fact(remaining_bits, start_j)) % MOD
                    start_j += K
            
            if binary_R[i] == '1':
                popcount_so_far += 1
        
        # Check R itself
        if popcount_so_far % K == 0:
            count = (count + 1) % MOD
        
        print(count)

    except EOFError:
        pass
    except ValueError:
        pass

if __name__ == "__main__":
    solve_final_optimized_time()

